/*--------------------------------------------------------------------------------------
  Copyright (c) 2021 GX2505, Inc. All rights reserved

  File Name   : onewire.c
  Author      : GXCAS
  Date        : 2021-06-24
  Version     : 0.1
  Description : configure the STM32 as a 1-wire master

  Modification History :
  Date      By       Version    Change Description
  ====================================================================================
  21/06/24  GXCAS  0.1        Original
--------------------------------------------------------------------------------------*/

#include "globle.h"

/*---------------------------------------------------------------------
  Function    : onewire_init
  Description : initialize 1-wire configuration
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void onewire_init(void)
{
	
	GPIO_InitTypeDef pin_cfg;
  // enable peripheral clock
  RCC_APB2PeriphClockCmd(ONEWIRE_CLK, ENABLE);
	
  // configure I/O port of DQ
  
  pin_cfg.GPIO_Pin   = ONEWIRE_PIN;
  pin_cfg.GPIO_Mode  = GPIO_Mode_Out_OD;	  // open-drain output
  pin_cfg.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(ONEWIRE_PORT, &pin_cfg);
	
	GPIO_SetBits(ONEWIRE_PORT, ONEWIRE_PIN);  // default HIGH
}

/*---------------------------------------------------------------------
  Function    : onewire_pulldown            
  Description : pull down the 1-wire bus
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void onewire_pulldown(void)
{
  GPIO_ResetBits(ONEWIRE_PORT, ONEWIRE_PIN);
}

/*---------------------------------------------------------------------
  Function    : onewire_release            
  Description : release the 1-wire bus
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void onewire_release(void)
{
  GPIO_SetBits(ONEWIRE_PORT, ONEWIRE_PIN);
}

/*---------------------------------------------------------------------
  Function    : onewire_receive
  Description : receive 1 bit of data from the bus
  Parameter   : none
  Return      : data received
---------------------------------------------------------------------*/
u8 onewire_receive(void)
{
	return GPIO_ReadInputDataBit(ONEWIRE_PORT, ONEWIRE_PIN);
}

/*---------------------------------------------------------------------
  Function    : onewire_reset             
  Description : initialization sequence (reset pulse + presence pulse)
  Parameter 1 : 1-wire communication speed
  Return      : if presence pulse is detected
---------------------------------------------------------------------*/
u8 onewire_reset(u8 spd)
{
  u8 rxd = 1;
  
	if(spd == SPD_STD) {
		onewire_pulldown();
		delay_560us();	// tRST
		onewire_release();
		delay_70us();		// tMSP
		rxd = onewire_receive();
		delay_560us();	// tRST
	}
	else {
		onewire_pulldown();
		delay_70us();		// tRST
		onewire_release();
		delay_8us();		// tMSP
		rxd = onewire_receive();
		delay_70us();		// tRST
	}

  return ((rxd == 0) ? ACK : NACK);
}

/*---------------------------------------------------------------------
  Function    : onewire_read_bit             
  Description : read one bit from the 1-wire bus
  Parameter 1 : 1-wire communication speed
  Return      : received bit
---------------------------------------------------------------------*/
u8 onewire_read_bit(u8 spd)
{
  u8 rxd = 0;
  
	if(spd == SPD_STD) {
		onewire_pulldown();	
		delay_8us();	// tRL
		onewire_release();
		delay_7us();	// tMSR - tRL
		rxd = onewire_receive();
		delay_85us();	// tSLOT - tMSR
	}
	else {
		onewire_pulldown();
		delay_1us();	// tRL
		onewire_release();
		delay_1us();	// tMSR - tRL
		rxd = onewire_receive();
		delay_18us();	// tSLOT - tMSR
	}
  return rxd;
}

/*---------------------------------------------------------------------
  Function    : onewire_read             
  Description : read one byte from the 1-wire bus
  Parameter 1 : 1-wire communication speed
  Return      : received byte
---------------------------------------------------------------------*/
u8 onewire_read(u8 spd)
{
  u8 dat = 0;
  u8 rxd = 0;
  u8 i;
  for(  i = 0; i < 8; i ++) {
    dat = dat >> 1;
    rxd = onewire_read_bit(spd);
    dat = dat | ((rxd == 0) ? 0x00 : 0x80);
  }
  
  return dat;
}

/*---------------------------------------------------------------------
  Function    : onewire_write_bit             
  Description : write one bit to the 1-wire bus
  Parameter 1 : 1-wire communication speed
  Parameter 2 : bit to be transmitted
  Return      : none
---------------------------------------------------------------------*/
void onewire_write_bit(u8 spd, u8 txd)
{
  if(txd == 0) {
		if(spd == SPD_STD) {
			onewire_pulldown();
			delay_70us();	// tW0L
			onewire_release();
			delay_30us();	// tSLOT - tW0L
		}
		else {
			onewire_pulldown();
			delay_8us();	// tW0L
			onewire_release();
			delay_12us();	// tSLOT - tW0L
		}
	}
  else {
		if(spd == SPD_STD) {
			onewire_pulldown();
			delay_8us();	// tW1L
			onewire_release();
			delay_92us();	// tSLOT - tW1L
		}
		else {
			onewire_pulldown();
			delay_1us();	// tW1L
			onewire_release();
			delay_19us();	// tSLOT - tW1L
		}
	}
}

/*---------------------------------------------------------------------
  Function    : onewire_write             
  Description : write one byte to the 1-wire bus
  Parameter 1 : 1-wire communication speed
  Parameter 2 : byte to be transmitted
  Return      : none
---------------------------------------------------------------------*/
void onewire_write(u8 spd, u8 dat)
{
  u8 txd = 0;
  u8 i;
	
  for(  i = 0; i < 8; i ++) {
    txd = dat & 0x01; // LSB first
    dat = dat >> 1;
    onewire_write_bit(spd, txd);
  }
}
