/*--------------------------------------------------------------------------------------
  Copyright (c) 2021 GXCAS, Inc. All rights reserved

  File Name   : gx2431.c
  Author      : gxcas
  Date        : 2021-06-24
  Version     : 0.1
  Description : driver for GX2431

  Modification History :
  Date      By       Version    Change Description
  ====================================================================================
  21/06/24  gxcas     0.1        Original
--------------------------------------------------------------------------------------*/

#include "globle.h"

/*---------------------------------------------------------------------
  Function    : gx28e17_read_rom             
  Description : execute READ_ROM command
  Parameter 1 : 1-wire communication speed
  Return      : ID number of current device (0=no device was found)
---------------------------------------------------------------------*/
uint64_t gx28e17_read_rom(u8 spd)
{
  u8  rxd = 0;
  uint64_t id  = 0;
  u8 i;
	
  if(onewire_reset(spd) == ACK) {
		onewire_write(spd, ROM_READ);
    for(  i = 0; i < 8; i ++) {
      rxd = onewire_read(spd);
			printf("%02X",rxd);
      id  = id | (((uint64_t) rxd) << (8 * i));
    }
  }
  else
    printf("Read ROM : No device on the bus.\r\n");
  
  return id;
}

 



/*---------------------------------------------------------------------
  Function    : gx28e17_match_rom             
  Description : execute MATCH_ROM command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : ID number of selected device
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_match_rom(u8 spd, uint64_t id)
{
  u8 txd = 0;
  u8 i;
	
  if(onewire_reset(spd) == ACK) {
    onewire_write(spd, ROM_MATCH);
    for(  i = 0; i < 8; i ++) {
      txd = (u8) (id >> (8 * i));
      onewire_write(spd, txd);
    }
    return DONE;
  }
  else {
    printf("Match ROM : No device on the bus.\r\n");
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_match_rom_od             
  Description : execute MATCH_ROM_OVERDRIVE command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : ID number of selected device
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_match_rom_od(u8 spd, uint64_t id)
{
  u8 txd = 0;
  u8 i;
	
  if(onewire_reset(spd) == ACK) {
    onewire_write(spd, ROM_MATCH_OD);
    for(  i = 0; i < 8; i ++) {
      txd = (u8) (id >> (8 * i));
      onewire_write(SPD_OD, txd);
    }
    return DONE;
  }
  else {
    printf("Match ROM (od) : No device on the bus.\r\n");
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_search_rom             
  Description : execute search algorithm
  Parameter 1 : 1-wire communication speed
  Parameter 2 : ID number array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
u8 gx28e17_search_rom(u8 spd, uint64_t * ids)
{
  u8  num     = 0;   // number of device found
  u8  pos     = 0;   // current bit position
  u8  dir     = 0;   // search direction
  u8  val     = 0;   // current bit
  u8  val_cmp = 0;   // complement of current bit
  u8  brn_new = 0;   // the bit position where a new branch is taken
  u8  brn_chk = 0;   // branch checkpoint
  uint64_t cur_id  = 0;   // current ID
  uint64_t reg_id  = 0;   // last ID found
  
  do {
    if(onewire_reset(spd) == ACK) {
      brn_new = 0;
      cur_id  = 0;
      onewire_write(spd, ROM_SEARCH);
      for(pos = 1; pos < 65; pos ++) {
        cur_id  = cur_id >> 1;
        val     = onewire_read_bit(spd);
        val_cmp = onewire_read_bit(spd);
        /*
          00 : There are both 0s and 1s in the current bit position of the participating ROM numbers => discrepancy
          01 : There are only 0s        in the current bit position of the participating ROM numbers
          10 : There are only 1s        in the current bit position of the participating ROM numbers
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last ROM number found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((u8) (reg_id >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
							printf("Search ROM : The device discovered is removed from the 1-wire bus during the search. (bit %d)\r\n", pos);
							return 0;
						}
        cur_id = cur_id | ((dir == 0) ? 0x0000000000000000 : 0x8000000000000000);
        onewire_write_bit(spd, dir);
      }
      brn_chk  = brn_new;
      reg_id   = cur_id;
      ids[num] = cur_id;
      num ++;
    }
    else {
      printf("Search ROM : No device on the bus.\r\n");
      return 0;
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx28e17_skip_rom             
  Description : execute SKIP_ROM command
  Parameter 1 : 1-wire communication speed
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_skip_rom(u8 spd)
{
  if(onewire_reset(spd) == ACK) {
    onewire_write(spd, ROM_SKIP);
    return DONE;
  }
  else {
    printf("Skip ROM : No device on the bus.\r\n");
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_skip_rom_od             
  Description : execute SKIP_ROM_OVERDRIVE command
  Parameter 1 : 1-wire communication speed
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_skip_rom_od(u8 spd)
{
  if(onewire_reset(spd) == ACK) {
    onewire_write(spd, ROM_SKIP_OD);
    return DONE;
  }
  else {
    printf("Skip ROM (od) : No device on the bus.\r\n");
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_resume_rom             
  Description : execute RESUME_ROM command
  Parameter 1 : 1-wire communication speed
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_resume_rom(u8 spd)
{
  if(onewire_reset(spd) == ACK) {
    onewire_write(spd, ROM_RESUME);
    return DONE;
  }
  else {
    printf("Resume ROM : No device on the bus.\r\n");
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_write_with_stop             
  Description : execute WRITE_DATA_WITH_STOP command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : I2C slave address
  Parameter 3 : I2C write length
  Parameter 4 : I2C write data array
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_write_with_stop(u8 spd, u8 addr, u8 wlen, u8 * wdat)
{
	u8  st  = 0;
  u8  wst = 0;
	u8  cnt = 0;
  uint16_t crc = 0;
  u8 i;
	
  // transmit 1-wire packet
  onewire_write(spd, DEV_WR_STP);
  onewire_write(spd, addr);
  onewire_write(spd, wlen);
  for(  i = 0; i < wlen; i ++)
    onewire_write(spd, wdat[i]);
  
  // crc
  crc = gx28e17_generate_crc(DEV_WR_STP, addr, wlen, wdat, 0);
  onewire_write(spd, ((u8)  crc));
  onewire_write(spd, ((u8) (crc >> 8)));
  
  // poll
  while(onewire_read_bit(spd) != 0) {
    delay_100us();
		cnt ++;
		if(cnt > POLL_MAX) {
			printf("Write with stop : Exceed query limit.\r\n");
			return ERROR;
    }
  }
  
  // read return bytes
  st  = onewire_read(spd);
  wst = onewire_read(spd);
  switch(st) {
    case ERR_STA : {
      printf("Write with stop : Invalid start condition.\r\n");
      return ERROR;
    }
    
    case ERR_ADR : {
      printf("Write with stop : I2C slave device does not acknowledge the I2C address.\r\n");
      return ERROR;
    }
    
    case ERR_CRC : {
      printf("Write with stop : 1-Wire packet data does not match the corresponding CRC-16.\r\n");
      return ERROR;
    }
    
    default : break;
  }
  if(wst == 0)
    return DONE;  // Done correctly
  else {
    printf("Write with stop : Byte %d is not acknowledged by the I2C slave device.\r\n", wst);
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_write_no_stop             
  Description : execute WRITE_DATA_NO_STOP command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : I2C slave address
  Parameter 3 : I2C write length
  Parameter 4 : I2C write data array
  Return      : finish flag
---------------------------------------------------------------------*/
//
u8 gx28e17_write_no_stop(u8 spd, u8 addr, u8 wlen, u8 * wdat)
{
  u8  st  = 0;
  u8  wst = 0;
	u8  cnt = 0;
  uint16_t crc = 0;
  u8 i;
	
  // transmit 1-wire packet
  onewire_write(spd, DEV_WR);//0x5A  --Write data no stop
  onewire_write(spd, addr);//ַ
  onewire_write(spd, wlen);//
  for(  i = 0; i < wlen; i ++)//0x02
    onewire_write(spd, wdat[i]);
  
  // crc
  crc = gx28e17_generate_crc(DEV_WR, addr, wlen, wdat, 0);
  onewire_write(spd, ((u8)  crc));
  onewire_write(spd, ((u8) (crc >> 8)));
  
  // poll---ʵַʽ֣ȴһԤʱ䣬BUSY pin״̬ȡ0
  while(onewire_read_bit(spd) != 0) {
    delay_100us();
		cnt ++;
		if(cnt > POLL_MAX) {
			printf("Write no stop : Exceed query limit.\r\n");
			return 0;
    }
  }
  
  // read return bytes
  st  = onewire_read(spd);
  wst = onewire_read(spd);
  switch(st) {
    case ERR_STA : {//0x08
      printf("Write no stop : Invalid start condition.\r\n");
      return ERROR;
    }
    
    case ERR_ADR : {//0x02
      printf("Write no stop : I2C slave device does not acknowledge the I2C address.\r\n");
      return ERROR;
    }
    
    case ERR_CRC : {//0x01
      printf("Write no stop : 1-Wire packet data does not match the corresponding CRC-16.\r\n");
      return ERROR;
    }
    
    default : break;
  }
  if(wst == 0)
    return DONE;  // Done correctly
  else {
    printf("Write no stop : Byte %d = %02X is not acknowledged by the I2C slave device.\r\n", wst,wst);
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_write_only_no_stop             
  Description : execute WRITE_DATA_ONLY command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : I2C write length
  Parameter 3 : I2C write data array
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_write_only_no_stop(u8 spd, u8 wlen, u8 * wdat)
{
  u8  st  = 0;
  u8  wst = 0;
	u8  cnt = 0;
  uint16_t crc = 0;
  u8 i;
	
  // transmit 1-wire packet
  onewire_write(spd, DEV_WR_OLY);
  onewire_write(spd, wlen);
  for( i = 0; i < wlen; i ++)
    onewire_write(spd, wdat[i]);
  
  // crc
  crc = gx28e17_generate_crc(DEV_WR_OLY, 0, wlen, wdat, 0);
  onewire_write(spd, ((u8)  crc));
  onewire_write(spd, ((u8) (crc >> 8)));
  
  // poll
  while(onewire_read_bit(spd) != 0) {
    delay_100us();
		cnt ++;
		if(cnt > POLL_MAX) {
			printf("Write only no stop : Exceed query limit.\r\n");
			return ERROR;
    }
  }
  
  // read return bytes
  st  = onewire_read(spd);
  wst = onewire_read(spd);
  switch(st) {
    case ERR_STA : {
      printf("Write only no stop : Invalid start condition.\r\n");
      return ERROR;
    }
    
    case ERR_ADR : {
      printf("Write only no stop : I2C slave device does not acknowledge the I2C address.\r\n");
      return ERROR;
    }
    
    case ERR_CRC : {
      printf("Write only no stop : 1-Wire packet data does not match the corresponding CRC-16.\r\n");
      return ERROR;
    }
    
    default : break;
  }
  if(wst == 0)
    return DONE;  // Done correctly
  else {
    printf("Write only no stop : Byte %d is not acknowledged by the I2C slave device.\r\n", wst);
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_write_only_with_stop             
  Description : execute WRITE_DATA_ONLY_WITH_STOP command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : I2C write length
  Parameter 3 : I2C write data array
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_write_only_with_stop(u8 spd, u8 wlen, u8 * wdat)
{
  u8  st  = 0;
  u8  wst = 0;
	u8  cnt = 0;
  uint16_t crc = 0;
  u8 i ;
	
  // transmit 1-wire packet
  onewire_write(spd, DEV_WR_OLY_STP);
  onewire_write(spd, wlen);
  for(  i = 0; i < wlen; i ++)
    onewire_write(spd, wdat[i]);
  
  // crc
  crc = gx28e17_generate_crc(DEV_WR_OLY_STP, 0, wlen, wdat, 0);
  onewire_write(spd, ((u8)  crc));
  onewire_write(spd, ((u8) (crc >> 8)));
  
  // poll
  while(onewire_read_bit(spd) != 0) {
    delay_100us();
		cnt ++;
		if(cnt > POLL_MAX) {
			printf("Write only with stop : Exceed query limit.\r\n");
			return ERROR;
    }
  }
  
  // read return bytes
  st  = onewire_read(spd);
  wst = onewire_read(spd);
  switch(st) {
    case ERR_STA : {
      printf("Write only with stop : Invalid start condition.\r\n");
      return ERROR;
    }
    
    case ERR_ADR : {
      printf("Write only with stop : I2C slave device does not acknowledge the I2C address.\r\n");
      return ERROR;
    }
    
    case ERR_CRC : {
      printf("Write only with stop : 1-Wire packet data does not match the corresponding CRC-16.\r\n");
      return ERROR;
    }
    
    default : break;
  }
  if(wst == 0)
    return DONE;  // Done correctly
  else {
    printf("Write only with stop : Byte %d is not acknowledged by the I2C slave device.\r\n", wst);
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_read_with_stop             
  Description : execute READ_DATA_WITH_STOP command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : I2C slave address
  Parameter 3 : I2C read length
  Parameter 4 : I2C read data array
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_read_with_stop(u8 spd, u8 addr, u8 rlen, u8 * rdat)
{
  u8  st  = 0;
	u8  cnt = 0;
  uint16_t crc = 0;
  u8 i;
	
  // transmit 1-wire packet
  onewire_write(spd, DEV_RD_STP);
  onewire_write(spd, addr);
  onewire_write(spd, rlen);
  
  // crc
  crc = gx28e17_generate_crc(DEV_RD_STP, addr, 0, 0, rlen);//ݵCRC16
  onewire_write(spd, ((u8)  crc));
  onewire_write(spd, ((u8) (crc >> 8)));
	
  // poll
  while(onewire_read_bit(spd) != 0) {
    delay_100us();
		cnt ++;
		if(cnt > POLL_MAX) {
			printf("Read with stop : Exceed query limit.\r\n");
			return ERROR;
    }
  }
	
  // read return bytes
  st  = onewire_read(spd);
  switch(st) {
    case ERR_STA : {
      printf("Read with stop : Invalid start condition.\r\n");
      return ERROR;
    }
    
    case ERR_ADR : {
      printf("Read with stop : I2C slave device does not acknowledge the I2C address.\r\n");
      return ERROR;
    }
    
    case ERR_CRC : {
      printf("Read with stop : 1-Wire packet data does not match the corresponding CRC-16.\r\n");
      return ERROR;
    }
    
    default : break;
  }
	
	for(  i = 0; i < rlen; i ++)
    rdat[i] = onewire_read(spd);
  return DONE;  // Done correctly
}

/*---------------------------------------------------------------------
  Function    : gx28e17_write_read_with_stop             
  Description : execute WRITE_READ_DATA_WITH_STOP command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : I2C slave address
  Parameter 3 : I2C write length
  Parameter 4 : I2C write data array
  Parameter 5 : I2C read length
  Parameter 6 : I2C read data array
  Return      : finish flag
---------------------------------------------------------------------*/
u8 gx28e17_write_read_with_stop(u8 spd, u8 addr, u8 wlen, u8 * wdat, u8 rlen, u8 * rdat)
{
  u8  st  = 0;
  u8  wst = 0;
	u8  cnt = 0;
  uint16_t crc = 0;
  u8 i;
	
  // transmit 1-wire packet
  onewire_write(spd, DEV_WR_RD_STP);
  onewire_write(spd, addr);
  onewire_write(spd, wlen);
  for(  i = 0; i < wlen; i ++)
    onewire_write(spd, wdat[i]);
  onewire_write(spd, rlen);
  
  // crc
  crc = gx28e17_generate_crc(DEV_WR_RD_STP, addr, wlen, wdat, rlen);
  onewire_write(spd, ((u8)  crc));
  onewire_write(spd, ((u8) (crc >> 8)));
  
  // poll
  while(onewire_read_bit(spd) != 0) {
    delay_100us();
		cnt ++;
		if(cnt > POLL_MAX) {
			printf("Write read with stop : Exceed query limit.\r\n");
			return ERROR;
    }
  }
  
  // read return bytes
  st  = onewire_read(spd);
  wst = onewire_read(spd);
  switch(st) {
    case ERR_STA : {
      printf("Write read with stop : Invalid start condition.\r\n");
      return ERROR;
    }
    
    case ERR_ADR : {
      printf("Write read with stop : I2C slave device does not acknowledge the I2C address.\r\n");
      return ERROR;
    }
    
    case ERR_CRC : {
      printf("Write read with stop : 1-Wire packet data does not match the corresponding CRC-16.\r\n");
      return ERROR;
    }
    
    default : break;
  }
  if(wst == 0) {
    for(  i = 0; i < rlen; i ++)
      rdat[i] = onewire_read(spd);
    return DONE;  // Done correctly
  }
  else {
    printf("Write read with stop : Byte %d is not acknowledged by the I2C slave device.\r\n", wst);
    return ERROR;
  }
}

/*---------------------------------------------------------------------
  Function    : gx28e17_write_config             
  Description : execute WRITE_CONFIGURATION command
  Parameter 1 : 1-wire communication speed
  Parameter 2 : configuration byte
  Return      : none
---------------------------------------------------------------------*/
void gx28e17_write_config(u8 spd, u8 cfg)
{
  onewire_write(spd, DEV_WR_CFG);
  onewire_write(spd, cfg);
}

/*---------------------------------------------------------------------
  Function    : gx28e17_read_config             
  Description : execute READ_CONFIGURATION command
  Parameter 1 : 1-wire communication speed
  Return      : configuration byte
---------------------------------------------------------------------*/
u8 gx28e17_read_config(u8 spd)
{
  u8 cfg = 0;
  
  onewire_write(spd, DEV_RD_CFG);
  cfg = onewire_read(spd);
  
  return cfg;
}

/*---------------------------------------------------------------------
  Function    : gx28e17_enable_sleep             
  Description : execute ENABLE_SLEEP_MODE command
  Parameter 1 : 1-wire communication speed
  Return      : none
---------------------------------------------------------------------*/
void gx28e17_enable_sleep(u8 spd)
{
  onewire_write(spd, DEV_EN_SLP);
}

/*---------------------------------------------------------------------
  Function    : gx28e17_read_revision             
  Description : execute READ_DEVICE_REVISION command
  Parameter 1 : 1-wire communication speed
  Return      : revision byte
---------------------------------------------------------------------*/
u8 gx28e17_read_revision(u8 spd)
{
  u8 rev = 0;
  
  onewire_write(spd, DEV_RD_REV);
  rev = onewire_read(spd);
  
  return rev;
}

/*---------------------------------------------------------------------
  Function    : gx28e17_generate_crc             
  Description : generate CRC-16 of 1-wire packet
  Parameter 1 : device command
  Parameter 2 : I2C slave address
  Parameter 3 : I2C write length
  Parameter 4 : I2C write data array
  Parameter 5 : I2C read length
  Return      : calculated CRC-16
---------------------------------------------------------------------*/
uint16_t gx28e17_generate_crc(u8 dcmd, u8 addr, u8 wlen, u8 * wdat, u8 rlen)
{
  u8  dat_in  = 0; // data input bit
  u8  crc_in  = 0; // crc  input bit
  uint16_t crc_reg = 0; // crc  register
  u8 i;
	u8 j;
	
  for(  i = 0; i < 8; i ++) {
    dat_in  = (dcmd >> i) & 0x01;
    crc_in  = ((u8) (crc_reg & 0x0001)) ^ dat_in;
    crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x0000);
  }
  
  if((dcmd == DEV_WR_STP) || (dcmd == DEV_WR) || (dcmd == DEV_WR_RD_STP) || (dcmd == DEV_RD_STP)) {
    for(  i = 0; i < 8; i ++) {
      dat_in  = (addr >> i) & 0x01;
      crc_in  = ((u8) (crc_reg & 0x0001)) ^ dat_in;
      crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x0000);
    }
  }
  
  if((dcmd == DEV_WR_STP) || (dcmd == DEV_WR) || (dcmd == DEV_WR_RD_STP) || (dcmd == DEV_WR_OLY) || (dcmd == DEV_WR_OLY_STP)) {
    for(  i = 0; i < 8; i ++) {
      dat_in  = (wlen >> i) & 0x01;
      crc_in  = ((u8) (crc_reg & 0x0001)) ^ dat_in;
      crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x0000);
    }
    for(  i = 0; i < wlen; i ++) {
      for(  j = 0; j < 8; j ++) {
        dat_in  = (wdat[i] >> j) & 0x01;
        crc_in  = ((u8) (crc_reg & 0x0001)) ^ dat_in;
        crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x0000);
      }
    }
  }
  
  if((dcmd == DEV_WR_RD_STP) || (dcmd == DEV_RD_STP)) {
    for(  i = 0; i < 8; i ++) {
      dat_in  = (rlen >> i) & 0x01;
      crc_in  = ((u8) (crc_reg & 0x0001)) ^ dat_in;
      crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x0000);
    }
  }
  
  return (~crc_reg);
}
