/*--------------------------------------------------------------------------------------
  Copyright (c) 2023 GXCAS, Inc. All rights reserved

  File Name   : gx1832_onewire.c
  File Type   : source file
  Author      : zhengkl
  Date        : 2023-04-11
  Version     : 0.1
  Description : implement the underlying timing of 1-wire bus

  Modification History :
  Date      By       Version    Change Description
  ====================================================================================
  23/04/11  zhengkl  0.1        Original
--------------------------------------------------------------------------------------*/

/*--------------------------------------------------------------------|
|                     !!!!!!!!!! NOTE !!!!!!!!!!                      |
|                                                                     |
|  The DQ port MUST be configured as open-drain output AT ALL TIMES   |
|--------------------------------------------------------------------*/

#include "gx1832.h"

/*---------------------------------------------------------------------
  Function    : gx1832_onewire_reset
  Description : initialization sequence (reset pulse + presence pulse)
  Parameter   : none
  Return      : if presence pulse is detected
---------------------------------------------------------------------*/
uint8_t gx1832_onewire_reset (void)
{
  uint8_t rxd = 1;
  
  // transmit reset pulse
	gx1832_gpio_pulldown();
  gx1832_gpio_delay(GX1832_TIME_RST);
	gx1832_gpio_release();
  
  // detect presence pulse
  gx1832_gpio_delay(GX1832_TIME_MSP);
	rxd = gx1832_gpio_sample();
  gx1832_gpio_delay(GX1832_TIME_RST - GX1832_TIME_MSP);

  return ((rxd == 0) ? GX1832_ACK : GX1832_NACK);
}

/*---------------------------------------------------------------------
  Function    : gx1832_onewire_read_bit
  Description : read one bit from the 1-wire bus
  Parameter   : none
  Return      : received bit
---------------------------------------------------------------------*/
uint8_t gx1832_onewire_read_bit (void)
{
  uint8_t rxd = 0;
  
  // start read slot
	gx1832_gpio_pulldown();	
  gx1832_gpio_delay(GX1832_TIME_RL);
	gx1832_gpio_release();
  
  // sample the 1-wire bus
	gx1832_gpio_delay(GX1832_TIME_MSR - GX1832_TIME_RL);
	rxd = gx1832_gpio_sample();
  gx1832_gpio_delay(GX1832_TIME_SLT - GX1832_TIME_MSR);
  
  // recovery time
  gx1832_gpio_delay(GX1832_TIME_REC);
  
  return rxd;
}

/*---------------------------------------------------------------------
  Function    : gx1832_onewire_write_bit
  Description : write one bit to the 1-wire bus
  Parameter 1 : bit to be transmitted
  Return      : none
---------------------------------------------------------------------*/
void gx1832_onewire_write_bit (uint8_t txd)
{
  if(txd == 0) {
		// start write-zero slot
    gx1832_gpio_pulldown();
    gx1832_gpio_delay(GX1832_TIME_W0L);
		gx1832_gpio_release();
	}
  else {
		// start write-one slot
    gx1832_gpio_pulldown();
    gx1832_gpio_delay(GX1832_TIME_W1L);
		gx1832_gpio_release();
    gx1832_gpio_delay(GX1832_TIME_SLT - GX1832_TIME_W1L);
	}
  
  // recovery time
  gx1832_gpio_delay(GX1832_TIME_REC);
}

/*---------------------------------------------------------------------
  Function    : gx1832_onewire_read_byte
  Description : read one byte from the 1-wire bus
  Parameter   : none
  Return      : received byte
---------------------------------------------------------------------*/
uint8_t gx1832_onewire_read_byte (void)
{
  uint8_t dat = 0;
  uint8_t rxd = 0;
  
  for(uint8_t i = 0; i < 8; i ++) {
    dat = dat >> 1;
    rxd = gx1832_onewire_read_bit();
    dat = dat | ((rxd == 0) ? 0x00 : 0x80);
  }
  
  return dat;
}

/*---------------------------------------------------------------------
  Function    : gx1832_onewire_write_byte
  Description : write one byte to the 1-wire bus
  Parameter 1 : byte to be transmitted
  Return      : none
---------------------------------------------------------------------*/
void gx1832_onewire_write_byte (uint8_t dat)
{
  uint8_t txd = 0;
  
  for(uint8_t i = 0; i < 8; i ++) {
    txd = dat & 0x01; // LSB first
    dat = dat >> 1;
    gx1832_onewire_write_bit(txd);
  }
}
